from pyparsing import Literal

from codeable_detectors.basic_detectors import AtLeastOneFileMatchesDetector
from codeable_detectors.evidences import LinkEvidence, FailedEvidence, ComponentEvidence
from codeable_detectors.js.jsDetectors import detect_js_import, JSCallSyncAsyncDetector
from codeable_detectors.pyparsing_patterns import ID, round_braces_block
from codeable_detectors.utils import update_keyword_args


# detect use of 'redis - a node.js redis client'
# https://www.npmjs.com/package/redis
class NodeJSRedisAPIClientCalls(AtLeastOneFileMatchesDetector):
    def __init__(self, redis_client_var):
        super().__init__()
        self.redisClientVar = redis_client_var
        self.file_endings = ["js"]

    def detect_in_context(self, ctx, **kwargs):
        # we use only ID here instead of listing all possible redis commands
        # (there are so many, see: https://redis.io/commands)
        matches = ctx.matches_pattern(Literal(self.redisClientVar) +
                                      Literal(".") + ID + round_braces_block + Literal(";"))
        if matches:
            return LinkEvidence(matches)
        return FailedEvidence("node JS redis API calls not found")


class NodeJSRedisDBLink(AtLeastOneFileMatchesDetector):
    def __init__(self):
        super().__init__()
        self.file_endings = ["js"]

    def detect_in_context(self, ctx, **kwargs):
        matches = []
        is_detected = False
        sync_async_list_types = []

        import_matches = detect_js_import(ctx, "redis")
        if not import_matches:
            return FailedEvidence("JS redis import not found")

        for import_match in import_matches:
            matches.append(import_match)

            # we detect use of importVar in the original context ctx!
            redis_create_client_matches = ctx.matches_pattern(ID + Literal("=") +
                                                              Literal(import_match.kwargs["importVar"]) +
                                                              Literal(".") + Literal("createClient"))
            for redisCreateClientMatch in redis_create_client_matches:
                matches.append(redisCreateClientMatch)
                redis_client_var = redisCreateClientMatch.text[:redisCreateClientMatch.text.find("=")].strip()

                sync_async_redis_calls_detector = JSCallSyncAsyncDetector(NodeJSRedisAPIClientCalls(redis_client_var))
                # we detect use of redis_client_var in the original context ctx!
                call_evidence = sync_async_redis_calls_detector.detect_in_context(ctx, **kwargs)
                if call_evidence.has_succeeded():
                    matches.extend(call_evidence.matches)
                    sync_async_list_types.extend(call_evidence.link_types)
                    is_detected = True

        if not is_detected:
            return FailedEvidence("node JS redis client creation or API usage not found")

        link_types = ["resp"]
        sync_async_list_type = _dedup_sync_async_link_types_list(sync_async_list_types)
        if sync_async_list_type:
            link_types.append(sync_async_list_type)

        return LinkEvidence(matches).set_properties(detector_link_types=link_types,
                                                    detector_technology_types="redis", kwargs=kwargs)


class NodeJSRedisDBComponent(NodeJSRedisDBLink):
    def detect_in_context(self, ctx, **kwargs):
        evidence = super().detect_in_context(ctx, **kwargs)
        if not evidence.has_succeeded():
            return evidence

        options = update_keyword_args({'guessed_name': None}, kwargs)
        name = options["guessed_name"] + " Redis DB"
        return ComponentEvidence(evidence.matches).set_properties(detector_name=name,
                                                                  detector_component_types="redisDB",
                                                                  detector_link_types=evidence.link_types,
                                                                  detector_technology_types="redis", kwargs=kwargs)


def _dedup_sync_async_link_types_list(sync_async_list_types):
    link_type = None
    if "syncAsyncConnector" in sync_async_list_types:
        link_type = "syncAsyncConnector"
    elif "synchronousConnector" in sync_async_list_types:
        if "asynchronousConnector" in sync_async_list_types:
            link_type = "syncAsyncConnector"
        else:
            link_type = "synchronousConnector"
    elif "asynchronousConnector" in sync_async_list_types:
        link_type = "asynchronousConnector"
    return link_type
